#!/bin/bash

# Constants
INSTALL_DIR="/opt/flashprogrammer"
VENV_DIR="$INSTALL_DIR/venv"
APP_SCRIPT_PATH="."
DESKTOP_FILE_PATH="/usr/share/applications/flashprogrammer.desktop"
TEMP_INSTALL_DIR="/tmp/flashprogrammer_install"
TEMP_VENV_DIR="$TEMP_INSTALL_DIR/venv"
FLASH_C_NAME="flashprogrammer"
FPGA_BIN_NAME="flashprogrammerFPGA.bin"

# Step 1: Prepare temporary installation directory
rm -rf "$TEMP_INSTALL_DIR"
mkdir -p "$TEMP_INSTALL_DIR"
if [ $? -ne 0 ]; then
  echo "Failed to create temporary installation directory: $TEMP_INSTALL_DIR"
  exit 1
fi

# Step 2: Copy application files to temporary directory
cp "$APP_SCRIPT_PATH"/*.py "$TEMP_INSTALL_DIR/"
cp "$APP_SCRIPT_PATH"/*.png "$TEMP_INSTALL_DIR/"
cp "$APP_SCRIPT_PATH"/"$FLASH_C_NAME" "$TEMP_INSTALL_DIR/"
cp "$APP_SCRIPT_PATH"/"$FPGA_BIN_NAME" "$TEMP_INSTALL_DIR/"

if [ $? -ne 0 ]; then
  echo "Failed to copy application files to: $TEMP_INSTALL_DIR"
  exit 1
fi

# Step 3: Set up a Python virtual environment in the temporary directory
python3 -m venv "$TEMP_VENV_DIR"
if [ $? -ne 0 ]; then
  echo "Failed to create virtual environment in: $TEMP_VENV_DIR"
  exit 1
fi

# Activate virtual environment and install dependencies
source "$TEMP_VENV_DIR/bin/activate"
if [ $? -ne 0 ]; then
  echo "Failed to activate temporary virtual environment."
  exit 1
fi

pip install --upgrade pip
pip install -r requirements.txt
if [ $? -ne 0 ]; then
  echo "Failed to install dependencies in temporary virtual environment."
  exit 1
fi

deactivate

# Step 4: Copy the prepared installation directory to the final location
sudo rm -rf "$INSTALL_DIR"
sudo cp -r "$TEMP_INSTALL_DIR" "$INSTALL_DIR"
if [ $? -ne 0 ]; then
  echo "Failed to copy installation files to: $INSTALL_DIR"
  exit 1
fi

# Clean up temporary installation directory
rm -rf "$TEMP_INSTALL_DIR"

# Step 5: Create a desktop file for the application
sudo bash -c "cat <<EOL > \"$DESKTOP_FILE_PATH\"
[Desktop Entry]
Version=1.0
Name=Flash Programmer
Comment=Flash Programmer GUI Utility
Exec=$VENV_DIR/bin/python $INSTALL_DIR/main.py
Icon=$INSTALL_DIR/icon.png
Terminal=false
Type=Application
Categories=Utility;
Path=$INSTALL_DIR
EOL"

if [ $? -ne 0 ]; then
  echo "Failed to create desktop entry: $DESKTOP_FILE_PATH"
  exit 1
fi

# Step 6: Set permissions and ownership
sudo chmod +x "$INSTALL_DIR/main.py"
sudo chmod +x "$INSTALL_DIR/$FLASH_C_NAME"
sudo chmod +x "$VENV_DIR/bin/python"
sudo chmod 644 "$DESKTOP_FILE_PATH"

# Step 7: Inform user
echo "Flash Programmer installed successfully!"
echo "You can launch it from your application menu."
