import os
import logging
import signal
import subprocess
import queue
import threading
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
from logging.handlers import RotatingFileHandler
from enum import Enum


# Constants
FLASH_C_PROGRAM_NAME = "flashprogrammer"
CONFIG_DIR = os.path.expanduser("~/.flashprogrammer")
LOG_FILE = os.path.join(CONFIG_DIR, "logs/output.log")
ERROR_LOG_FILE = os.path.join(CONFIG_DIR, "logs/error.log")
LAST_DIR_FILE = os.path.join(CONFIG_DIR, "last_dir.txt")

# Ensure directories exist
os.makedirs(os.path.dirname(LOG_FILE), exist_ok=True)

# Configure logging
log_formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger()
logger.setLevel(logging.DEBUG)

file_handler = RotatingFileHandler(LOG_FILE, maxBytes=10240, backupCount=5)
file_handler.setFormatter(log_formatter)
logger.addHandler(file_handler)

error_handler = RotatingFileHandler(ERROR_LOG_FILE, maxBytes=10240, backupCount=5)
error_handler.setFormatter(log_formatter)
logger.addHandler(error_handler)


class MessageType(Enum):
    STATUS = "status"
    LOG = "log"
    ERROR = "error"
    CONTROL = "control"


class FlashProgrammerGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("Flash Programmer")
        self.root.geometry("600x400")

        self.selected_file = None
        self.last_dir = self.get_last_dir()
        self.queue = queue.Queue()
        self.process = None

        # Create Widgets
        file_frame = tk.Frame(self.root)
        file_frame.pack(pady=10, fill=tk.X)

        self.select_button = tk.Button(file_frame, text="Select File", command=self.select_file)
        self.select_button.pack(side=tk.LEFT, padx=5)

        self.file_path_label = tk.Label(file_frame, text="No file selected", fg="red", anchor="w")
        self.file_path_label.pack(side=tk.LEFT, padx=10, fill=tk.X, expand=True)

        self.flash_button = tk.Button(self.root, text="Flash", command=self.start_flash_thread)
        self.flash_button.pack(pady=10)

        self.status_label = tk.Label(self.root, text="Waiting for input...", wraplength=400)
        self.status_label.pack(pady=10)

        self.log_text = scrolledtext.ScrolledText(self.root, wrap=tk.WORD, height=15, width=70)
        self.log_text.pack(pady=10)

        # Handle window close
        self.root.protocol("WM_DELETE_WINDOW", self.on_close)

        # Set initial status
        self.update_status("Waiting for file input...")

    def get_last_dir(self):
        """Read the last accessed directory from last_dir.txt."""
        if os.path.exists(LAST_DIR_FILE):
            with open(LAST_DIR_FILE, "r") as file:
                return file.read().strip()
        return os.path.expanduser("~")  # Default to the user's home directory

    def save_last_dir(self, directory):
        """Save the last accessed directory to last_dir.txt."""
        with open(LAST_DIR_FILE, "w") as file:
            file.write(directory)

    def update_status(self, message):
        """Update the status label with the given message."""
        self.status_label.config(text=message)

    def select_file(self):
        file_path = filedialog.askopenfilename(initialdir=self.last_dir, title="Select File")
        if file_path:
            self.selected_file = file_path
            self.last_dir = os.path.dirname(file_path)  # Update the last directory
            self.save_last_dir(self.last_dir)  # Save it for future use
            self.file_path_label.config(text=file_path, fg="black")
            self.update_status("File selected. Ready to flash.")
        else:
            self.selected_file = None
            self.file_path_label.config(text="No file selected", fg="red")
            self.update_status("Waiting for file input...")

    def start_flash_thread(self):
        if not self.selected_file:
            messagebox.showerror("Error", "No file selected.")
            return

        # Disable UI and update status
        self.set_buttons_state(tk.DISABLED)
        self.update_status("Flashing in progress...")

        # Start Flashing in a New Thread
        threading.Thread(target=self.flash_file, daemon=True).start()

    def flash_file(self):
        """Execute the flashing process."""
        try:
            self.queue.put((MessageType.STATUS, "Flashing in progress..."))
            logger.info("Starting flash process.")

            self.process = subprocess.Popen(
                [f"./{FLASH_C_PROGRAM_NAME}", "-P1", self.selected_file],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True
            )

            # Start threads for stdout and stderr
            threading.Thread(target=self.read_pipe, args=(self.process.stdout, MessageType.LOG), daemon=True).start()
            threading.Thread(target=self.read_pipe, args=(self.process.stderr, MessageType.ERROR), daemon=True).start()

            # Wait for the process to complete
            self.process.wait()

            # Process completion status
            if self.process.returncode == 0:
                self.queue.put((MessageType.STATUS, "Flashing completed successfully."))
                logger.info("Flashing completed successfully.")
            else:
                self.queue.put((MessageType.STATUS, "Flashing failed. Check logs for details."))
                logger.error("Flashing failed.")
        except Exception as e:
            logger.exception("Error during flashing process.")
            self.queue.put((MessageType.ERROR, f"Error during flashing: {str(e)}"))
        finally:
            # Re-enable UI
            self.queue.put((MessageType.CONTROL, "enable_buttons"))

    def read_pipe(self, pipe, message_type):
        """Read lines from a process pipe and add them to the queue."""
        try:
            for line in iter(pipe.readline, ""):
                self.queue.put((message_type, line.strip()))
                if message_type == MessageType.ERROR:
                    logger.error(line.strip())
                else:
                    logger.info(line.strip())
        finally:
            pipe.close()

    def set_buttons_state(self, state):
        """Enable or disable buttons."""
        self.select_button.config(state=state)
        self.flash_button.config(state=state)

    def process_queue(self):
        """Process messages from the queue."""
        while not self.queue.empty():
            message_type, content = self.queue.get_nowait()
            if message_type == MessageType.STATUS:
                self.update_status(content)
            elif message_type == MessageType.LOG:
                self.log_text.insert(tk.END, content + "\n")
                self.log_text.see(tk.END)
            elif message_type == MessageType.ERROR:
                self.log_text.insert(tk.END, f"ERROR: {content}\n")
                self.log_text.see(tk.END)
            elif message_type == MessageType.CONTROL and content == "enable_buttons":
                self.set_buttons_state(tk.NORMAL)
                self.update_status("Waiting for file input...")

        self.root.after(100, self.process_queue)

    def on_close(self):
        """Handle closing of the application, including process termination."""
        if self.process and self.process.poll() is None:
            # If the flash process is running, send SIGINT
            try:
                self.queue.put((MessageType.LOG, "Sending SIGINT to flash process..."))
                self.process.send_signal(signal.SIGINT)
                self.queue.put((MessageType.LOG, "SIGINT sent to flash process. Waiting for it to terminate..."))

                # Give the process up to 2 seconds to terminate
                try:
                    self.process.wait(timeout=2)
                    self.queue.put((MessageType.LOG, "Flash process terminated gracefully."))
                except subprocess.TimeoutExpired:
                    self.queue.put((MessageType.LOG, "Flash process did not terminate; killing it now."))
                    self.process.kill()
                    self.process.wait()
                    self.queue.put((MessageType.LOG, "Flash process forcibly killed."))
            except Exception as e:
                self.queue.put((MessageType.ERROR, f"Failed to terminate flash process: {str(e)}"))
        else:
            self.queue.put((MessageType.LOG, "No flash process running or already terminated."))

        # Destroy the GUI
        self.root.destroy()


if __name__ == "__main__":
    root = tk.Tk()
    app = FlashProgrammerGUI(root)
    root.after(100, app.process_queue)  # Start queue processing
    root.mainloop()
