#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>
#include <time.h>
#include <wiringPi.h>


#define FILECRC     0x1
#define ERASE       0x2
#define WRITE       0x4
#define READ        0x20
#define FLASHCRC    0x8
#define FLASHCRCEND 0x10


// GLBDEF.H
#define ERR_NO				0
#define ERR_FILEOPEN		1
#define ERR_FILESIZE		3
#define ERR_SW				4
#define ERR_UNSUPP			5
#define ERR_FCARDFSIMM		6
#define ERR_NODEVICE		7
#define ERR_MEMORY			8
#define ERR_COMMANDLINE		9

#define ERR_VPP			20
#define ERR_ERASE		21
#define ERR_WRITE		22
#define ERR_CRC			23
#define ERR_READ		24


//#define TRUE			1
//#define FALSE			0

#define ENABLE_SPI_LOG	0


// DEVICE.H
#define READARRAY         0xFFFF
#define INTELIGENTID      0x9090
#define CLEARSTATUS       0x5050
#define BLOCKERASE        0x2020
#define BLOCKERASECONFIRM 0xD0D0
#define WORDWRITE         0x4040


#define INTEL      0x89
#define I28F008    0xA2
#define I28F020    0xBD

#define I28F008SIZE    0x100000L
#define I28F020SIZE    0x040000L


// HW.H
#define OUT8255 0
#define IN8255  1


// SPI
#define SPEED 7000000


// GLBDEF.H
typedef unsigned short WORD;
typedef unsigned char BYTE;
typedef BYTE BOOL;
typedef unsigned long LWORD;


// FLASH.H
typedef struct {
	LWORD adr[130];		// Endadresse
	WORD size[130];
	BYTE block[130];
	BYTE cs[130];
	BYTE chip[130];
	BYTE nsektor;
	WORD crc[130];
	int type;
	LWORD devicesize;
} SECTOR;

// FLASH.C
// defines for SIMM-SIZE         // PD 6  2  1
#define SIMNO       0x00          //    0  0  0
#define SIM256K     0x01          //    0  0  1
#define SIM512K     0x02          //    0  1  0
#define SIM1M       0x03          //    0  1  1
#define SIM2M       0x20          //    1  0  0
#define SIM4M       0x21          //    1  0  1
#define SIM8M       0x22          //    1  1  0
#define SIM16M      0x23          //    1  1  1

SECTOR sektor;

BYTE *pBuffer;           // Buffer
BYTE *pBufferSave;       // Buffer

BYTE SimType;



typedef struct {
	unsigned int Adress : 32;
} pinArrayAdress;

union pinAdress {
	char pinBytes[4];
	pinArrayAdress pA;
};

typedef struct {
	unsigned int Data : 16;
	unsigned int Header : 16;
} pinArrayData;

union pinData {
	char pinBytes[4];
	pinArrayData pD;
};

typedef struct {
	unsigned int CE : 4;
	unsigned int OE : 1;
	unsigned int WE_L : 1;
	unsigned int WE_H : 1;
	unsigned int REG : 1;
	unsigned int RST : 1;
	unsigned int VPP : 1;
	unsigned int VCC : 1;
	unsigned int BEEP : 1;
	unsigned int DATOE : 1;
	unsigned int SPI_SEND1 : 1;
	unsigned int SPI_SEND2 : 1;
	unsigned int SPI_SEND3 : 1;
	unsigned int Header : 16;
} pinArrayConfig;

union pinConfig {
	char pinBytes[4];
	pinArrayConfig pC;
};

union pinWrite {
	char pinBytes[4];
	pinArrayConfig pC;
};


//union pinDataOld defaultPins;
//union pinDataOld pinValues;

union pinAdress defaultAdress;
union pinAdress pinAdressValues;

union pinData defaultData;
union pinData pinDataValues;

union pinConfig defaultConfig;
union pinConfig pinConfigValues;

union pinWrite pinWriteValues;

int fd;



void delay_ms(int ms) {
	usleep(ms * 1000);
}

/* ---------------------------------------------------------------------
 * SPI
 * -------------------------------------------------------------------*/ 

int spiOpen(unsigned spiChan, unsigned spiBaud, unsigned spiFlags) {
	int fd;
	char spiMode;
	char spiBits = 8;
	char dev[32];
	
	spiMode = spiFlags & 3;
	spiBits = 8;
	
	sprintf(dev, "/dev/spidev0.%d", spiChan);
	
	if ((fd = open(dev, O_RDWR)) < 0) {
		return -1;
	}
	
	if (ioctl(fd, SPI_IOC_WR_MODE, &spiMode) < 0) {
		close(fd);
		return -2;
	}
	
	if (ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &spiBits) < 0) {
		close(fd);
		return -3;
	}
	
	if (ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &spiBaud) < 0) {
		close(fd);
		return -4;
	}
	
	return fd;
}

int spiClose(int fd) {
	return close(fd);
}

int spiRead(int fd, unsigned speed, char *buf, unsigned count) {
	int err;
	struct spi_ioc_transfer spi;
	
	memset(&spi, 0, sizeof(spi));
	
	spi.tx_buf = (unsigned) NULL;
	spi.rx_buf = (unsigned) buf;
	spi.len = count;
	spi.speed_hz = speed;
	spi.delay_usecs = 0;
	spi.bits_per_word = 8;
	spi.cs_change = 0;
	
	err = ioctl(fd, SPI_IOC_MESSAGE(1), &spi);
	
	return err;
}

int spiWrite(int fd, unsigned speed, char *buf, unsigned count) {
	int err;
	struct spi_ioc_transfer spi;
	
	#if ENABLE_SPI_LOG == 1
		printf("pinValues: %02X", buf[0]);
		printf("%02X", buf[1]);
		printf("%02X", buf[2]);
		printf("%02X\n", buf[3]);
	#endif
	
	
	memset(&spi, 0, sizeof(spi));
	
	spi.tx_buf = (unsigned) buf;
	spi.rx_buf = (unsigned) NULL;
	spi.len = count;
	spi.speed_hz = speed;
	spi.delay_usecs = 0;
	spi.bits_per_word = 8;
	spi.cs_change = 0;
	
	err = ioctl(fd, SPI_IOC_MESSAGE(1), &spi);
	
	return err;
}


int configureFpga(void) {
	int fpga_config_handle;
	char buf[128];
	int bytes = 0;
	int ret = ERR_NO;
	
	
	// reset FPGA
	digitalWrite(3, LOW);
	delay_ms(10);
	digitalWrite(3, HIGH);
	delay_ms(10);
	
	
	if ((fpga_config_handle = open("./flashprogrammerFPGA.bin", S_IREAD)) == -1) {
		printf("Failed to open FPGA configuration file\n");
		return ERR_FILEOPEN;
	} 
	
	while(1) {
		memset(buf, 0, 128);
		bytes = read(fpga_config_handle, buf, 128);
		
		if (bytes == -1) {
			printf("Error while reading FPGA configuration file\n");
			ret = ERR_FILEOPEN;
			break;
		} else {
			
			if (bytes == 0) {
				printf("Finished reading FPGA configuration file\n");
				ret = ERR_NO;
				break;
			}
			
			spiWrite(fd, SPEED, buf, bytes);
		}
	}	
	
	delay_ms(1000);
	
	return ret;
}

/* ---------------------------------------------------------------------
 * HW
 * -------------------------------------------------------------------*/

BOOL gCardInitialized = FALSE;
unsigned long SaveAddr;
BYTE  directionDataOut;

int hwInit(void) {
	
	#if ENABLE_SPI_LOG == 1
		printf("spiWrite: hwInit -> ");
	#endif
	// set default values for PINS
	spiWrite(fd, SPEED, defaultAdress.pinBytes, 4);
	spiWrite(fd, SPEED, defaultData.pinBytes, 4);
	spiWrite(fd, SPEED, defaultConfig.pinBytes, 4);
	
	directionDataOut = FALSE;
	gCardInitialized = TRUE;
	
	return ERR_NO;
}

void hwCleanup(void) {
	
	#if ENABLE_SPI_LOG == 1
		printf("spiWrite: hwCleanup -> ");
	#endif
	spiWrite(fd, SPEED, defaultAdress.pinBytes, 4);
	spiWrite(fd, SPEED, defaultData.pinBytes, 4);
	spiWrite(fd, SPEED, defaultConfig.pinBytes, 4);
	
	gCardInitialized = FALSE;
}

BYTE hwGetType(void) {
	char res[2];
	
	pinConfigValues.pC.SPI_SEND1 = 0;
	pinConfigValues.pC.SPI_SEND2 = 1;
	pinConfigValues.pC.SPI_SEND3 = 0;
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwRequestData -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
	
	pinConfigValues.pC.SPI_SEND1 = 1;
	pinConfigValues.pC.SPI_SEND2 = 0;
	pinConfigValues.pC.SPI_SEND3 = 1;
	
	// readchannel CD
	#if ENABLE_SPI_LOG == 1
		printf("Read CD\n");
	#endif
	
	spiRead(fd, SPEED, res, 2);
	
	//printf("hwGetType = 0x%02X\n", res[1]);
	
	return (res[1] ^ 0xFF);
}

void hwPowerOn(BYTE on) {
	int i1, i2;
	
	if (on != 0) {
		pinConfigValues.pC.VCC = 0;
		pinConfigValues.pC.RST = 0;
	} else {
		pinConfigValues.pC.VCC = 1;
	}
	
	#if ENABLE_SPI_LOG == 1
		printf("spiWrite: hwPowerOn -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
	
	for(i1=0;i1<2000;i1++) {
		for(i2=0;i2<20000;i2++);
	}
}

void hwVPPOn(BYTE on) {
	int i1, i2;
	
	if (on != 0) {
		pinConfigValues.pC.VPP = 1;
	} else {
		pinConfigValues.pC.VPP = 0;
	}
	
	#if ENABLE_SPI_LOG == 1
		printf("spiWrite: hwVPPOn -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
	
	for(i1=0;i1<2000;i1++) {
		for(i2=0;i2<20000;i2++);
	}
}

void hwRequestData(void) {
	pinConfigValues.pC.SPI_SEND1 = 0;
	pinConfigValues.pC.SPI_SEND2 = 1;
	pinConfigValues.pC.SPI_SEND3 = 0;
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwRequestData -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
	
	pinConfigValues.pC.SPI_SEND1 = 1;
	pinConfigValues.pC.SPI_SEND2 = 0;
	pinConfigValues.pC.SPI_SEND3 = 1;
}

void hwVPPA9On(BYTE on) {
	// set Pin VppA9 depending on on
	printf("NI: hwVPPA9On\n");
}

void hwCE(BYTE on, BYTE ce) {
	
	switch (ce) {
		case 0:
			if (on != 0) {
				pinConfigValues.pC.CE = 0xF;
			} else {
				pinConfigValues.pC.CE = 0xE;
			}
			break;
		case 1:
			if (on != 0) {
				pinConfigValues.pC.CE = 0xF;
			} else {
				pinConfigValues.pC.CE = 0xD;
			}
			break;
		case 2:
			if (on != 0) {
				pinConfigValues.pC.CE = 0xF;
			} else {
				pinConfigValues.pC.CE = 0xB;
			}
			break;
		case 3:
			if (on != 0) {
				pinConfigValues.pC.CE = 0xF;
			} else {
				pinConfigValues.pC.CE = 0x7;
			}
			break;
		case 10:
			if (on != 0) {
				pinConfigValues.pC.CE = 0xF;
			} else {
				pinConfigValues.pC.CE = 0xC;
			}
	}

	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwCE -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
}

void hwWE(BYTE on) {
	if (on != 0) {
		pinConfigValues.pC.WE_H = 1;
		pinConfigValues.pC.WE_L = 1;
	} else {
		if (directionDataOut == FALSE) {
			printf("FATAL ERROR hwWE directionDataOut\n");
			exit(1);
		}
		pinConfigValues.pC.WE_H = 0;
		pinConfigValues.pC.WE_L = 0;
	}
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwWE -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
}

void hwOE(BYTE on) {
	if (on != 0) {
		pinConfigValues.pC.OE = 1;
	} else {
		if (directionDataOut != FALSE) {
			pinConfigValues.pC.DATOE = 0;
			directionDataOut = FALSE;
		}
		pinConfigValues.pC.OE = 0;
	}
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwOE -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
}

void hwSetAddr(LWORD adr) {
	SaveAddr = adr;
	pinAdressValues.pA.Adress = (unsigned long) SaveAddr;
	
	// add adress header
	pinAdressValues.pinBytes[3] |= 0x30;
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwSetAddr -> ");
	#endif
	spiWrite(fd, SPEED, pinAdressValues.pinBytes, 4);
}

void hwSetAddrPlus1(void) {
	SaveAddr++;
	pinAdressValues.pA.Adress = (unsigned long) SaveAddr;
	
	// add adress header
	pinAdressValues.pinBytes[3] |= 0x30;
	
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwSetAddrPlus1 -> ");
	#endif
	spiWrite(fd, SPEED, pinAdressValues.pinBytes, 4);
}

void hwSet8255InOut(BYTE Inout) {
	if (Inout != OUT8255) {
		if (directionDataOut != FALSE) {
			pinConfigValues.pC.DATOE = 0;
			directionDataOut = FALSE;
		}
	} else {
		if (directionDataOut == FALSE) {
			pinConfigValues.pC.DATOE = 1;
			directionDataOut = TRUE;
		}
	}
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwSetInOut -> ");
	#endif
	spiWrite(fd, SPEED, pinConfigValues.pinBytes, 4);
}

void hwWriteWord(WORD data) {
	hwSet8255InOut(OUT8255);
	
	pinDataValues.pD.Data = data;
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwWriteWord -> ");
	#endif
	spiWrite(fd, SPEED, pinDataValues.pinBytes, 4);
	
	hwWE(0);
	hwWE(1);
}

void hwWriteWordFromBuffer(void) {
	unsigned long data;
	data = (*pBuffer) << 8;
	data |= *(pBuffer+1);
	pinDataValues.pD.Data = data;
	
	#if ENABLE_SPI_LOG == 1
	printf("spiWrite: hwWriteWordFromBuffer -> ");
	#endif
	spiWrite(fd, SPEED, pinDataValues.pinBytes, 4);
	
	//printf("WWFB: 0x%04X\n", data);
	
	hwWE(0);
	hwWE(1);
}

WORD hwReadWord(void) {
	char res[2];
	
	hwOE(0);
	//readChannel DataIn
	hwRequestData();
	#if ENABLE_SPI_LOG == 1
	printf("Read DataIn: ");
	#endif
	spiRead(fd, SPEED, res, 2);
	#if ENABLE_SPI_LOG == 1
	printf("0x%02X%02X\n", res[0], res[1]);
	//printf("0x%04X\n", (WORD)*res);
	#endif
	hwOE(1);
	
	//printf("RW: 0x%04X\n", (WORD)(res[1] | res[0] << 8));
	
	return (WORD)(res[1] | (res[0] << 8));
}

WORD hwReadWordWhenA9AtVpp(void) {
	unsigned long res;
	
	hwOE(0);
	hwVPPA9On(1);
	printf("Read DataIn (A9)\n");
	// readchannel DataIn
	hwVPPA9On(0);
	hwOE(1);
	
	return ((WORD) res);
}

BYTE hwReadByteL(void) {
	char res[2];
	
	hwOE(0);
	hwRequestData();
	#if ENABLE_SPI_LOG == 1
	printf("Read Byte L: ");
	#endif
	spiRead(fd, SPEED, res, 2);
	#if ENABLE_SPI_LOG == 1
	printf("0x%02X\n", res[1]);
	#endif
	hwOE(1);
	
	return ((BYTE)res[1]);
}

void hwReadWordToBuffer(void) {
	char val[2];
	
	hwOE(0);
	hwRequestData();
	#if ENABLE_SPI_LOG == 1
	printf("Read DataIn: ");
	#endif
	spiRead(fd, SPEED, val, 2);
	*pBuffer++ = val[0];
	*pBuffer++ = val[1];
	#if ENABLE_SPI_LOG == 1
	printf("0x%02X%02X\n", val[0], val[1]);
	#endif
	
	//printf("RWTB: 0x%04X\n", (WORD)(val[0] | val[1] << 8));
	
	hwOE(1);
	return;
}



/* ---------------------------------------------------------------------
 * DEVICE
 * -------------------------------------------------------------------*/

void define_IMC004FLKA(void) {
	int i;
	
	int block;
	int cs;
	int chip;
	LWORD adr;
	
	adr = 0;
	cs = 10;
	block = 0;
	chip = -1;
	
	for(i=0; i<128; i++) {
		adr += 0x4000L;
		
		if (i==0) adr = 0;
		
		if (i==0 || i==16 || i==32 || i==48 || i==64 || i==80 || i==96 || i==112) {
			chip++;
		}
		
		sektor.adr[i] = adr;
		sektor.chip[i] = chip;
		
		sektor.size[i] = 0x4000;
		sektor.block[i] = block;
		sektor.cs[i] = cs;
	}
	
	sektor.nsektor = 128;
	sektor.type = 1100;
	sektor.devicesize = 0x400000;
}


void define_IMC004FLSA(void) {
	int i;
	
	int block;
	int cs;
	int chip;
	LWORD adr;
	
	adr = 0;
	cs = 10;
	block = -1;
	chip = -1;
	
	for(i=0; i<128; i++) {
		adr += 0x4000L;
		
		if (i==0) adr = 0;
		
		if (i==64) chip++;
		
		if ((i & 0x3)==0) block ++;
		
		sektor.adr[i] = adr;
		sektor.chip[i] = chip;
		
		sektor.size[i] = 0x4000;
		sektor.block[i] = block;
		sektor.cs[i] = cs;
	}
	
	sektor.nsektor = 128;
	sektor.type = 1200;
	sektor.devicesize = 0x400000;
}

void definechips_i28F016(void) {
	int i;
	int block;
	int cs;
	LWORD adr;
	
	adr = 0;
	cs = -1;
	block = -1;
	
	for (i=0; i<64; i++) {
		adr += 0x4000L;
		
		if (i==0 || i==16 || i==32 || i==48) {
			adr = 0x0L;
			cs++;
		}
		
		if ((i&0x1) == 0) block++;
		
		sektor.adr[i] = adr;
		sektor.size[i] = 0x4000;
		sektor.chip[i] = 1;
		sektor.block[i] = block;
		sektor.cs[i] = cs;
	}
	
	sektor.nsektor = 64;
	sektor.type = 200;
	sektor.devicesize = 0x200000;
}

void define_ISM002FLK(void) {
	int i;
	int block;
	int cs;
	LWORD adr;
	
	adr = 0;
	cs = -1;
	block = 0;
	
	for (i=0; i<64; i++) {
		adr += 0x4000L;
		
		if (i==0 || i==16 || i==32 || i==48) {
			adr = 0x0L;
			cs++;
		}
		
		sektor.adr[i] = adr;
		sektor.size[i] = 0x4000;
		sektor.block[i] = block;
		sektor.cs[i] = cs;
		sektor.chip[i] = cs;
	}
	
	sektor.nsektor = 64;
	sektor.type = 100;
	sektor.devicesize = 0x200000;
}

void definechips_i29F160(void) {
	int i;
	int block;
	int cs;
	LWORD adr;
	
	adr = 0;
	cs = -1;
	block = 0;
	
	for (i=0; i<64; i++) {
		adr += 0x4000L;
		
		if (i==0 || i==16 || i==32 || i==48) {
			adr = 0x0L;
			cs++;
		}
		
		sektor.adr[i] = adr;
		sektor.size[i] = 0x4000;
		sektor.block[i] = block;
		sektor.cs[i] = cs;
		sektor.chip[i] = 1;
	}
	
	sektor.nsektor = 64;
	sektor.type = 300;
	sektor.devicesize = 0x200000;
}



/* ---------------------------------------------------------------------
 * CRC
 * -------------------------------------------------------------------*/
/******************* definitions for the 16-bit CRC function ****************/
const unsigned int crctab[256] = {
	0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7,
	0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF,
	0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6,
	0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE,
	0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485,
	0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D,
	0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4,
	0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC,
	0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823,
	0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B,
	0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12,
	0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A,
	0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41,
	0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49,
	0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70,
	0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78,
	0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F,
	0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067,
	0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E,
	0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256,
	0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D,
	0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
	0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C,
	0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634,
	0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB,
	0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3,
	0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A,
	0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92,
	0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9,
	0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1,
	0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8,
	0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0
	};
	
WORD crcval[10];

void crcinit(BYTE ind) {
	crcval[ind] = 0;
}

WORD crcresult(BYTE ind) {
	return crcval[ind];
}

void crc(BYTE data, BYTE ind) {
	crcval[ind] = crctab[(crcval[ind]>>8) & 0xFF] ^ (crcval[ind]<<8) ^ data;
}

/* ---------------------------------------------------------------------
 * FLASH
 * -------------------------------------------------------------------*/

int flashFileCRC(char* filename, WORD* filecrcsum) {
	int sek;
	int filelength;
	struct stat file_status;
	LWORD von;
	LWORD bis;
	LWORD w;
	
	int ret = ERR_NO;
	int handle;
	WORD co;
	WORD co1;
	
	if ((handle = open(filename, S_IREAD)) == -1) {
		ret = ERR_FILEOPEN;
	} else {
		int i;
		
		if (stat(filename, &file_status) < 0) {
			ret = ERR_FILEOPEN;
		}
		filelength = file_status.st_size;
		
		if (filelength != sektor.devicesize) {
			ret = ERR_FILESIZE;
		} else {
			sek = 0;
			crcinit(0);
			
			while (sek < sektor.nsektor && ret ==ERR_NO) {
				printf("FILE CRC %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
				
				von = sektor.adr[sek];
				bis = sektor.adr[sek]+sektor.size[sek]-1;
				
				pBuffer = pBufferSave;
				read(handle, pBuffer, 0x8000);
				pBuffer = pBufferSave;
				
				for (w=von; w<=bis; w++) {
					crc(*pBuffer, 0);
					
					pBuffer++;
					crc(*pBuffer, 0);
					pBuffer++;
				}
				
				sektor.crc[sek] = crcresult(0);
				sek++;
				
				//if (ret == ERR_NO) {
				//	ret = checkctrlc();
				//}
			}
		}
		
		close(handle);
	}
	
	if (ret == ERR_NO) {
		printf("FILE CRC 100%% \r");
	}
	
	*filecrcsum = crcresult(0);
	return (ret);
}




int flashGetIDCode(LWORD Manadr, LWORD Devadr, BYTE *Manufact, BYTE *Device) {
	hwSetAddr(Manadr);
	hwWriteWord(INTELIGENTID);
	
	*Manufact = hwReadWord() & 0xFF;
	
	hwSetAddr(Devadr);
	*Device = hwReadWord() & 0xFF;
	
	return ERR_NO;
}

int flashGetIDCode_29Fxxx(WORD *Manufact, WORD *Device) {
	hwSetAddr(0x555);
	hwWriteWord(0xAA);
	hwSetAddr(0x2AA);
	hwWriteWord(0x55);
	hwSetAddr(0x555);
	hwWriteWord(0x90);
	
	hwSetAddr(0);
	*Manufact = hwReadWordWhenA9AtVpp();
	hwSetAddr(1);
	*Device = hwReadWordWhenA9AtVpp();
	hwWriteWord(0xF0);
	
	return (ERR_NO);
}

int flashGetCardtype(void) {
	int ret;
	int i;
	BYTE Manufact;
	BYTE Device;
	BYTE ce = 10;
	
	LWORD chipsize;
	WORD devicesize;
	
	chipsize = 0;
	
	ret = ERR_UNSUPP;
	
	hwCE(0, ce);
	flashGetIDCode(0L, 2L, &Manufact, &Device);
	hwWriteWord(READARRAY);


	if (Manufact == INTEL) {
		if (Device == I28F008) chipsize = I28F008SIZE;
		if (Device == I28F020) chipsize = I28F020SIZE;
		
		if (chipsize != 0) {
			LWORD w;
			BYTE chips;
			BYTE ende;
			
			w = 0;
			chipsize <<= 1;
			chips = 0;
			ende = FALSE;
			
			while (ende == FALSE) {
				BYTE Device; // why declare this variable again here?
				
				flashGetIDCode(w, w+2, &Manufact, &Device);
				if (Manufact == INTEL) {
					chips++;
					
					w += chipsize;
					hwSetAddr(w);	// Adresse für Manufactcode einstellen
					
					if (hwReadWord() == ((WORD)(INTEL<<8)|INTEL)) ende=TRUE;
				} else {
					ende = TRUE;
				}
			}
			
			w = 0;	// Alle Chips auf READ-Memory stellen
			for(i=0; i<chips; i++) {
				hwSetAddr(w);
				hwWriteWord(0x0000);
				hwWriteWord(READARRAY);
				
				w += chipsize;
			}
			
			chipsize >>= 11;		// chipsize in KBYTE
			chips <<= 1;			// 2 chips / Word
			devicesize = chips * chipsize;
			printf("--- FLashCard  %d Chips a %d KByte ", chips, chipsize);
			printf("(%d KByte)\n", devicesize);
			
			if ((Device == I28F020) && (chips == 16)) {
				printf("--- iMC004FLKA\n");
				define_IMC004FLKA();	// 4 MB Flash-Card mit 28f020
				ret = ERR_NO;
			}
			
			if ((Device == I28F008) && (chips == 4)) {
				printf("--- iMC004FLSA (Serie 2)\n");
				define_IMC004FLSA();	// 4 MB Flash-Card mit 28f008
				ret = ERR_NO;
			}
			
		}
	}

	hwCE(1, ce);

	return (ret);
}

int flashGetSimmtype(void) {
	int ret;
	
	//BYTE SimType;
	BYTE Manufact = 0;
	BYTE Device = 0;
	
	//SimType = hwGetType();
	//SimType = 0x3;
	//printf("SimType = 0x%2X\n", SimType);
	
	hwCE(0, 0);
	flashGetIDCode(0L, 1L, &Manufact, &Device);
	hwWriteWord(0x0000);
	hwWriteWord(READARRAY);
	hwCE(1, 0);
	
	//printf("SIMM-Manufact: 0x%02X\n", Manufact);
	//printf("SIMM-Device: 0x%02X\n", Device);
	
	ret = ERR_UNSUPP;
	switch (SimType & 0x23) {
		case SIMNO:
			ret = ERR_NODEVICE;
			break;
		case SIM256K:
			printf("FLASH-SIMM 512K Byte");		// 128kW
			break;
		case SIM512K:
			printf("FLASH-SIMM 1024K Byte");	// 512kW
			break;
		case SIM1M:
			if (Manufact == 0x89 && Device == 0xA0) {
				definechips_i28F016();	// 1 Chip 2M Byte
				printf("FLASH-SIMM 1 Chip a 2048 KByte\n");
				printf("ALCATEL-SIMM\n");
				ret = ERR_NO;
			} else if (Manufact == 0x89 && Device == 0xBD) {
				define_ISM002FLK();		// 8 Chips a 256k Byte
				printf("FLASH-SIMM 8 Chips a 256 KByte\n");
				printf("iSM002FLKA\n");
				ret = ERR_NO;
			} else {
				WORD Manufact = 0;
				WORD Device = 0;
				hwCE(0, 0);
				flashGetIDCode_29Fxxx(&Manufact, &Device);
				hwCE(1, 0);
				if (Manufact == 0x0001 && Device == 0x22D8) {
					definechips_i29F160();	// 1 Chip a 2M Byte
					printf("FLASH-SIMM 1 Chip a 2 MByte\n");
					printf("i29F160\n");
					ret = ERR_NO;
				}
			}
			break;
		case SIM2M:
			printf("FLASH-SIMM 4M Byte");	// 2048kW
			break;
		case SIM4M:
			printf("FLASH-SIMM 8M Byte");	// 4096kW
			break;
		case SIM8M:
			printf("FLASH-SIMM 16M Byte");	// 8192kW
			break;
		case SIM16M:
			printf("FLASH-SIMM 32M Byte");	// 16384kW
			break;
		}
		
	
	return (ret);
}


int flashGetDevicetype(void) {
	//BYTE SimType;
	int ret;
	
	ret = ERR_NO;
	
	SimType = hwGetType();
	
	//printf("SimType = 0x%02X\n", SimType);
	
	if ((SimType & 0x81) == 0x81) {
		if (SimType == 0x81) {		// Flash-Card
			ret = flashGetCardtype();
		} else {
			ret = ERR_FCARDFSIMM;	// Flash-Card & Flash-Simm
		}
	} else {
		ret = flashGetSimmtype();
	}
	
	return (ret);
}

int flashReadBuffer(int type, BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD c;
	int i;
	
	switch(type) {
		case 100:	// FLASH-SIMM mit 28F010 OR 28F020
		case 200:
		case 300:
			hwCE(0, ce);
			
			pBuffer = pBufferSave;
			c = addr1;
			hwSetAddr(c);
			while (c <= addr2) {
				hwReadWordToBuffer();
				hwSetAddrPlus1();
				
				c++;
			}
			hwCE(1, ce);
			break;
		
		case 1100:
		case 1200:
			hwCE(0, ce);
			
			addr1 <<= 1;
			addr2 <<= 1;
			
			pBuffer = pBufferSave;
			c = addr1;
			hwSetAddr(c);
			while (c <= addr2) {
				hwReadWordToBuffer();
				hwSetAddrPlus1();
				hwSetAddrPlus1();
				
				c+=2;
			}
			hwCE(1, ce);
			break;
	}
	
	return ERR_NO;
}

int flashReadCRC(WORD* crcsum, BYTE boe) {
	int sek;
	LWORD von;
	LWORD bis;
	LWORD w;
	int ret = ERR_NO;
	
	crcinit(0);
	
	sek = 8;
	sek = 0;
	while(sek<sektor.nsektor && ret == ERR_NO) {
		printf("FLASH CRC %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
		
		von = sektor.adr[sek];
		bis = sektor.adr[sek]+sektor.size[sek]-1;
		flashReadBuffer(sektor.type, sektor.cs[sek], von, bis);
		
		pBuffer = pBufferSave;
		for (w=von; w<=bis; w++) {
			crc(*pBuffer, 0);
			
			pBuffer++;
			
			crc(*pBuffer, 0);
			
			pBuffer++;
		}
		
		if (boe != 0) {
			if (crcresult(0) != sektor.crc[sek]) {
				printf("\nCRC_SOLL=%04X CRC_IST=%04X\n", sektor.crc[sek], crcresult(1));
				ret = ERR_CRC;
			}
		}
		
		sek++;
		
		//if (ret == ERR_NO) {
		//	ret = checkctrlc();
		//}
	}
	
	if (ret == ERR_NO) {
		printf("FLASH CRC 100%%\r");
	}
	
	*crcsum = crcresult(0);
	
	return ret;
}


int flashRead(char * filename) {
	int sek;
	LWORD von;
	LWORD bis;
	LWORD w;
	int ret = ERR_NO;
	int handle;
	
	handle = creat(filename, S_IREAD | S_IWRITE);
	
	if (handle < 0) {
		printf("ERROR opening file\n");
		ret = ERR_FILEOPEN;
	}
	
	//printf("Anzahl Sektoren: %d\n", sektor.nsektor);
	
	sek = 0;
	while(sek<sektor.nsektor && ret==ERR_NO) {
		printf("--- FLASH READ %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
		
		von = sektor.adr[sek];
		bis = sektor.adr[sek] + sektor.size[sek]-1;
		flashReadBuffer(sektor.type, sektor.cs[sek], von, bis);
		
		pBuffer = pBufferSave;
		
		write(handle, pBuffer, 0x8000);
		
		sek++;
	}
	
	close(handle);
	return ret;
}

int flashWriteBuffer_V100(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD c;
	BYTE status;
	int i, i1;
	int ret;
	
	WORD data;
	WORD data1;
	
	hwCE(0, ce);
	
	pBuffer = pBufferSave;
	c = addr1;
	hwSetAddr(c);
	
	ret = ERR_NO;
	
	while((c<=addr2) && (ret==ERR_NO)) {
		i=0;
		
		data = (WORD)((*pBuffer) << 8) | (*(pBuffer + 1));
		//printf("data = %04X\n", data);
		data1 = data ^ 0xFFFF;
		//printf("data1 = %04X\n", data1);
		while(data != data1 && i++<100) {
			hwWriteWord(0x4040);
			hwWriteWordFromBuffer();
			hwWriteWord(0xC0C0);
			data1 = hwReadWord();
			
			//if (i > 1) {
			//	printf("i=%d; data=%04X; data1=%04X; addr=%04X\n", i, data, data1, c);
			//}
			
		}
		
		if (i >= 100) {
			ret = ERR_WRITE;
		}
		
		pBuffer += 2;
		hwSetAddrPlus1();
		c++;
	}
	
	hwSetAddr(addr1);
	hwWriteWord(0x0000);
	hwCE(1, ce);
	
	return (ret);
}

int flashWriteBuffer_V1100(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD c;
	BYTE status;
	int i, i1;
	int ret;
	
	WORD data;
	WORD data1;
	
	
	hwCE(0, ce);
	addr1 <<= 1;
	addr2 <<= 1;
	
	pBuffer = pBufferSave;
	c = addr1;
	hwSetAddr(c);
	
	ret = ERR_NO;
	while((c<=addr2) && (ret == ERR_NO)) {
		i=0;
		
		data = (WORD)((*pBuffer) << 8) | (*(pBuffer+1));
		data1 = data ^ 0xFFFF;
		while(data != data1 && i++ <100) {
			hwWriteWord(0x4040);
			hwWriteWordFromBuffer();
			hwWriteWord(0xC0C0);
			data1 = hwReadWord();
		}
		
		if (i >= 100) {
			ret = ERR_WRITE;
		}
		
		pBuffer += 2;
		hwSetAddrPlus1();
		hwSetAddrPlus1();
		c += 2;
	}
	
	hwSetAddr(addr1);
	hwWriteWord(0x0000);
	hwCE(1, ce);
	
	return (ret);
}

int flashWriteBuffer_V200(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD c;
	BYTE status;
	int i;
	int ret;
	
	hwCE(0, ce);
	
	pBuffer = pBufferSave;
	c = addr1;
	hwSetAddr(c);
	ret = ERR_NO;
	while((c <= addr2) && (ret == ERR_NO)) {
		hwWriteWord(CLEARSTATUS);
		
		hwWriteWord(WORDWRITE);
		hwWriteWordFromBuffer();
		
		i = 0;
		status = 0;
		while(((status & 0x80) == 0) && (i++ <1000)) {
			status = hwReadByteL();
			//printf("status = 0x%02X\n", status);
		}
		
		if (i >= 1000) {
			printf("WB_V200: timeout\n");
			ret = ERR_WRITE;
		}
		
		if (status & 0x10 != 0x0) {
			ret = ERR_WRITE;
		}
		
		if (status & 0x08 != 0x0) {
			ret = ERR_VPP;
		}
		
		pBuffer += 2;
		hwSetAddrPlus1();
		c++;
	}
	
	hwSetAddr(addr1);
	hwWriteWord(READARRAY);
	hwCE(1, ce);
	
	return (ret);
}

int flashWriteBuffer_V1200(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD c;
	WORD status;
	int i;
	int ret;
	
	unsigned long data;
	
	//printf("START WRITE\n");
	
	usleep(100);
	
	hwCE(0, ce);

	//printf("addr1=%08X\n", addr1);
	//printf("addr2=%08X\n", addr2);

	addr1 <<= 1;
	addr2 <<= 1;
	
	pBuffer = pBufferSave;
	c = addr1;
	hwSetAddr(c);
	ret = ERR_NO;
	
	//printf("addr1=%08X\n", addr1);
	//printf("addr2=%08X\n", addr2);
	
	// write_FPGA
	// Set FPGA into Write mode
	spiWrite(fd, SPEED, pinWriteValues.pinBytes, 4);
	// Send Start Adress
	pinAdressValues.pA.Adress = addr1;
	spiWrite(fd, SPEED, pinAdressValues.pinBytes, 4);
	// Send End Adress
	pinAdressValues.pA.Adress = addr2;
	spiWrite(fd, SPEED, pinAdressValues.pinBytes, 4);
	// write_FPGA
	while((c <= addr2) && (ret == ERR_NO)) {
		//hwWriteWord(CLEARSTATUS);
		
		//hwWriteWord(WORDWRITE);
		//hwWriteWordFromBuffer();
		
		//printf("Send data\n\n");
		
		data = (*pBuffer) << 8;
		data |= *(pBuffer+1);
		pinDataValues.pD.Data = data;
		spiWrite(fd, SPEED, pinDataValues.pinBytes, 2);
		
		
		//printf("%04X\n", data);
		
		//exit(1);
		
		/*
		i = 0;
		status = 0;
		while(((status & 0x8080) != 0x8080) && (i++ < 1000)) {
			status = hwReadWord();
			//printf("status: 0X%04X\n", status);
		}
		
		if (i > 1) {
			printf("i=%d\n", i);
		}
		
		//printf("\n");
		
		if (i >= 1000) {
			printf("V1200: i >= 1000\n");
			ret = ERR_WRITE;
		}
		
		if (status & 0x1010 != 0x0) {
			printf("V1200: status = 0x%02X\n", status);
			ret = ERR_WRITE;
		}
		
		if (status & 0x0808 != 0x0) {
			ret = ERR_VPP;
		}
		*/
		
		usleep(450);
		
		pBuffer += 2;
		//hwSetAddrPlus1();
		//hwSetAddrPlus1();
		c += 2;
	}
	
	hwSetAddr(addr1);
	hwWriteWord(READARRAY);
	hwCE(1, ce);
	
	return (ret);
}

int flashWriteBuffer_V300(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD addr;
	BYTE status;
	WORD *data;
	WORD counter;
	int ret;
	
	hwCE(0, ce);
	
	data = (WORD*) pBufferSave;
	addr = addr1;
	ret = ERR_NO;
	while((addr <= addr2) && (ret == ERR_NO)) {
		hwSetAddr(0x555);
		hwWriteWord(0xAA);
		hwSetAddr(0x2AA);
		hwWriteWord(0x55);
		hwSetAddr(0x555);
		hwWriteWord(0xA0);
		hwSetAddr(addr);
		hwWriteWord(*data);
		
		counter = 1000;
		while(1) {
			status = hwReadByteL();
			if ((0x80 & *data) == (0x80 & status)) {
				break;
			}
			--counter;
			if (0 == counter) {
				ret = ERR_WRITE;
				break;
			}
		}
		
		++data;
		++addr;
	}
	
	hwSetAddr(0);
	hwWriteWord(0xF0);
	
	hwCE(1, ce);
	
	return ret;
}


int flashWriteFromBinfile(char* filename) {
	int sek;
	LWORD von;
	LWORD bis;
	LWORD w;
	
	int ret = ERR_NO;
	int handle;
	WORD co;
	WORD co1;
	
	if ((handle = open(filename, S_IREAD)) == -1) {
		ret = ERR_FILEOPEN;
	} else {
		int i;
		
		printf("opened file: %s\n", filename);
		
		sek = 0;
		while(sek<sektor.nsektor && ret == ERR_NO) {
			printf("WRITE %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
			fflush(stdout);
			
			von = sektor.adr[sek];
			bis = sektor.adr[sek] + sektor.size[sek]-1;
			
			pBuffer = pBufferSave;
			if (read(handle, pBuffer, 0x8000) == (int)0x8000L) {
				pBuffer = pBufferSave;
			
				//printf("sektor_type: %d\n", sektor.type);
			
				switch(sektor.type) {
					case 100:
						ret = flashWriteBuffer_V100(sektor.cs[sek], von, bis);
						break;
					case 200:
						ret = flashWriteBuffer_V200(sektor.cs[sek], von, bis);
						break;
					case 300:
						ret = flashWriteBuffer_V300(sektor.cs[sek], von, bis);
						break;
					case 1100:
						ret = flashWriteBuffer_V1100(sektor.cs[sek], von, bis);
						break;
					case 1200:
						ret = flashWriteBuffer_V1200(sektor.cs[sek], von, bis);
						break;
					default:
						ret = ERR_SW;
				}
			} else {
				ret = ERR_FILESIZE;
			}
			
			sek++;
			
			//if (ret==ERR_NO) {
			//	ret = checkctrlc();
			//
		}
		close(handle);
	}
	if (ret == ERR_NO) {
		printf("WRITE 100%%\n");
	}
	
	return (ret);
}

int flashEraseChip_V100(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD w;
	WORD data;
	int i;
	int i1;
	BYTE ret;
	
	hwCE(0, ce);
	w = addr1;
	hwSetAddr(w);
	
	ret = ERR_NO;
	i = 0;
	data = 0;
	
	while(data != 0xFFFF && i++ < 3000 && w <= addr2) {
		hwWriteWord(0x2020);
		hwWriteWord(0x2020);
		
		delay_ms(5);
		
		data = 0xFFFF;
		while((data == 0xFFFF) && (w <= addr2)) {
			hwWriteWord(0xA0A0);
			
			data = hwReadWord();
			
			if (data == 0xFFFF) {
				hwSetAddrPlus1();
				w++;
			}
		}
	}
	
	hwSetAddr(addr1);
	hwWriteWord(0x0000);
	hwCE(1, ce);
	
	if (i >= 3000) {
		ret = ERR_ERASE;
	}
	
	return ret;
}

int flashEraseChip_V1100(BYTE ce, LWORD addr1, LWORD addr2) {
	LWORD w;
	WORD data;
	int i;
	int i1;
	BYTE ret;
	
	hwCE(0, ce);
	
	addr1 <<= 1;
	addr2 <<= 1;
	
	w = addr1;
	hwSetAddr(w);
	
	ret = ERR_NO;
	i = 0;
	data = 0;
	
	while(data != 0xFFFF && i++ <3000 && w <= addr2) {
		hwWriteWord(0x2020);
		hwWriteWord(0x2020);
		
		delay_ms(15);
		
		data = 0xFFFF;
		while((data == 0xFFFF) && (w <= addr2)) {
			hwWriteWord(0xA0A0);
			
			data = hwReadWord();
			
			if (data == 0xFFFF) {
				hwSetAddrPlus1();
				hwSetAddrPlus1();
				w += 2;
			}
		}
	}
	
	hwSetAddr(addr1);
	hwWriteWord(0x0000);
	hwCE(1, ce);
	
	if (i >= 3000) {
		ret = ERR_ERASE;
	}
	
	return ret;
}

int flashErase_V100(void) {
	int ret;
	LWORD w;
	WORD sek;
	LWORD adr;
	LWORD von;
	LWORD bis;
	
	BYTE cs;
	BYTE chip;
	
	ret = ERR_NO;
	
	chip = 0xFF;
	sek = 0;
	while((sek<sektor.nsektor) && (ret == ERR_NO)) {
		printf("ERASE %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
		
		von = sektor.adr[sek];
		cs = sektor.cs[sek];
		chip = sektor.chip[sek];
		
		while((sektor.chip[sek]==chip) && (sek<sektor.nsektor)) {
			bis = sektor.adr[sek]+sektor.size[sek]-1;
			sek++;
		}
		
		switch(sektor.type) {
			case 100:
				ret = flashEraseChip_V100(cs, von, bis);
				break;
			case 1100:
				ret = flashEraseChip_V1100(cs, von, bis);
				break;
		}
		
		//if (ret == ERR_NO) {
		//	ret = checkctrlc();
		//}
	}
	
	if (ret == ERR_NO) {
		printf("ERASE 100%%\n");
	}
	
	return ret;
}

BYTE flashEraseBlock_V200(BYTE ce, LWORD bladr) {
	BYTE status;
	clock_t timeout;
	
	hwSetAddr(bladr);
	hwCE(0, ce);
	
	hwWriteWord(CLEARSTATUS);
	
	hwWriteWord(BLOCKERASE);
	hwWriteWord(BLOCKERASECONFIRM);
	
	timeout = clock() + 2 * CLOCKS_PER_SEC;
	status = 0;
	
	//printf("cps: %ld\n", CLOCKS_PER_SEC);
	//printf("clock: %ld\n", clock());
	//printf("timeout: %ld\n", timeout);
	
	while(((status & 0x80) == 0) && (clock() <= timeout)) {
		delay_ms(250);
		status = hwReadByteL();
		//printf("status = 0x%02X\n", status);
		//printf("clock: %ld\n", clock());
		timeout -= 250 * 1000;
	}
	
	hwWriteWord(READARRAY);
	hwCE(1, ce);
	
	if (status & 0x08 != 0x0) {
		return ERR_VPP;
	}
	if (status & 0x30 != 0x0) {
		return ERR_NO;
	}
	
	return ERR_NO;
}

BYTE flashEraseBlock_V1200(BYTE ce, LWORD bladr) {
	WORD status;
	clock_t timeout;
	int i;
	
	hwCE(0, ce);
	bladr <<= 1;
	hwSetAddr(bladr);
	
	hwWriteWord(BLOCKERASE);
	hwWriteWord(BLOCKERASECONFIRM);
	
	timeout = clock() + 10 * CLOCKS_PER_SEC;
	status = 0;
	while(((status & 0x8080) != 0x8080) && (clock() <= timeout)) {
		delay_ms(125);
		status = hwReadWord();
		//printf("status = 0x%04X\n", status);
	}
	
	hwWriteWord(CLEARSTATUS);
	
	hwWriteWord(READARRAY);
	hwCE(1, ce);
	
	if (status & 0x0808 != 0x0) {
		return ERR_VPP;
	}
	if (status & 0x3030 != 0x0) {
		return ERR_ERASE;
	}
	
	return ERR_NO;
}

int flashErase_V200_V1200(void) {
	int ret;
	WORD sek;
	WORD block;
	LWORD adr;
	
	ret = ERR_NO;
	block = 0xFFFF;
	sek = 0;
	
	//printf("nsektor: %d\n", sektor.nsektor);
	
	while(sek<sektor.nsektor && ret==ERR_NO) {
		printf("ERASE %d%% \r", (int)(float)(((float)sek/(float)sektor.nsektor)*(float)100));
		
		if (block != sektor.block[sek]) {
			block = sektor.block[sek];
			
			switch(sektor.type) {
				case 200:
					ret = flashEraseBlock_V200(sektor.cs[sek], sektor.adr[sek]);
					break;
				case 1200:
					ret = flashEraseBlock_V1200(sektor.cs[sek], sektor.adr[sek]);
					break;
			}
			
			switch(ret) {
				case ERR_NO:
					break;
				case ERR_VPP:
					ret = ERR_VPP;
					printf("\nVPPERROR %d\n", sek);
					break;
				case ERR_ERASE:
					ret = ERR_ERASE;
					printf("\nERROR %d\n", sek);
					break;
			}
		}
		
		//printf("sektor: %d\n", sek);
		sek++;
		//if (ret==ERR_NO) {
		//	ret = checkctrlc();
		//}
	}
	
	if (ret == ERR_NO) {
		printf("ERASE 100%%\n");
	}
	return ret;
}

int flashEraseChip_V300(void) {
	printf("NI: flashEraseChip_V300\n");
	return ERR_SW;
}

int flashErase(void) {
	int ret = ERR_NO;
	
	switch(sektor.type) {
		case 100:					// FLASH-SIMM Chip 28f010 & 28f020
			ret = flashErase_V100();
			break;
		case 200:					// FLASH-SIMM Chip 28f008 & 28f016
			ret = flashErase_V200_V1200();
			break;
		case 300:					// FLASH-SIMM Chip 29f160
			ret = flashEraseChip_V300();
			break;
		case 1100:					// FLASH-Card Chip 28f010 & 28f020
			ret = flashErase_V100();
			break;
		case 1200:					// FLASH-Card Chip 28f008 & 28f016
			ret = flashErase_V200_V1200();
			break;
		default:
			ret = ERR_SW;
	}
	
	return ret;
}





int main(int argc, char *argv[]) {	
	int err = ERR_NO;
	char filename[1024];
	WORD job;
	WORD filecrcsum;
	WORD flashcrcsum;
	
	int cardType = 0;
	int fR = 0;
	int fW = 0;
	int fE = 0;
	
	
	defaultAdress.pA.Adress = 0x0CFFFFFF;
	defaultAdress.pinBytes[3] |= 0x30;

	pinAdressValues.pA.Adress = 0x0CFFFFFF;
	pinAdressValues.pinBytes[3] |= 0x30;
	
	defaultData.pD.Data = 0x0000;
	defaultData.pD.Header = 0x5000;

	pinDataValues.pD.Data = 0x0000;
	pinDataValues.pD.Header = 0x5000;
	
	defaultConfig.pC.CE = 0xF;
	defaultConfig.pC.OE = 1;
	defaultConfig.pC.WE_L = 1;
	defaultConfig.pC.WE_H = 1;
	defaultConfig.pC.REG = 1;
	defaultConfig.pC.RST = 1;
	defaultConfig.pC.VPP = 0;
	defaultConfig.pC.VCC = 1;
	defaultConfig.pC.BEEP = 0;
	defaultConfig.pC.DATOE = 0;
	defaultConfig.pC.SPI_SEND1 = 1;
	defaultConfig.pC.SPI_SEND2 = 0;
	defaultConfig.pC.SPI_SEND3 = 1;
	defaultConfig.pC.Header = 0x9000;
	//
	pinConfigValues.pC.CE = 0xF;
	pinConfigValues.pC.OE = 1;
	pinConfigValues.pC.WE_L = 1;
	pinConfigValues.pC.WE_H = 1;
	pinConfigValues.pC.REG = 1;
	pinConfigValues.pC.RST = 1;
	pinConfigValues.pC.VPP = 0;
	pinConfigValues.pC.VCC = 1;
	pinConfigValues.pC.BEEP = 0;
	pinConfigValues.pC.DATOE = 0;
	pinConfigValues.pC.SPI_SEND1 = 1;
	pinConfigValues.pC.SPI_SEND2 = 0;
	pinConfigValues.pC.SPI_SEND3 = 1;
	pinConfigValues.pC.Header = 0x9000;
	
	pinWriteValues.pC.CE = 0x0;
	pinWriteValues.pC.OE = 0;
	pinWriteValues.pC.WE_L = 0;
	pinWriteValues.pC.WE_H = 0;
	pinWriteValues.pC.REG = 0;
	pinWriteValues.pC.RST = 0;
	pinWriteValues.pC.VPP = 0;
	pinWriteValues.pC.VCC = 0;
	pinWriteValues.pC.BEEP = 0;
	pinWriteValues.pC.DATOE = 0;
	pinWriteValues.pC.SPI_SEND1 = 0;
	pinWriteValues.pC.SPI_SEND2 = 0;
	pinWriteValues.pC.SPI_SEND3 = 0;
	pinWriteValues.pC.Header = 0x0800;
	
	// Disable buffering for stdout
	setvbuf(stdout, NULL, _IONBF, 0);
	
	// Setup GPIOs
	if (wiringPiSetup() == -1) {
		printf("WiringPiSetup failed\n");
		return 1;
	}
	// Set program_b pin (active low async full-chip reset for the FPGA)
	pinMode(3, OUTPUT);
	digitalWrite(3, HIGH);
	
	// SPI
	fd = spiOpen(0, SPEED, 0);
	if (fd < 0) {
		printf("fd=%d\n", fd);
		return 1;
	}
	
	
	// Command line parameter handling
	if (argc<2) {
		err = ERR_COMMANDLINE;
	}
	
	if (err == ERR_NO) {
		if (argv[1][0] != '-') {
			err = ERR_COMMANDLINE;
		}
	}
	
	if (err == ERR_NO) {
		switch (argv[1][1]) {
			case 'P':
				if (argc != 3) {
					err = ERR_COMMANDLINE;
				}
				
				if (argv[1][2] == '1') {
					job = FILECRC | ERASE | WRITE | FLASHCRC;
				} else if (argv[1][2] == '2') {
					job = FILECRC | WRITE | FLASHCRC;
				} else {
					err = ERR_COMMANDLINE;
				}
				
				if (err == ERR_NO) {
					strcpy(filename, argv[2]);
					printf("FILE: %s\n", filename);
					fflush(stdout);
				}
				break;
				
			case 'W':
				if (argc != 3) {
					err = ERR_COMMANDLINE;
				} else {
					job = WRITE;
					strcpy(filename, argv[2]);
					printf("FILE: %s\n", filename);
				}
				break;
				
			case 'E':
				if (argc != 2) {
					err = ERR_COMMANDLINE;
				}
				job = ERASE;
				break;
			
			case 'C':
				if (argc != 2) {
					err = ERR_COMMANDLINE;
				}
				job = FLASHCRCEND;
				break;
				
			case 'F':
				if (argc != 3) {
					err = ERR_COMMANDLINE;
				} else {
					job = FILECRC;
					strcpy(filename, argv[2]);
					printf("FILE: %s\n", filename);
				}
				break;
			
			case 'R':
				if (argc != 3) {
					err = ERR_COMMANDLINE;
				} else {
					job = FILECRC | READ | FLASHCRC;
					strcpy(filename, argv[2]);
					printf("FILE: %s\n", filename);
				}
				break;
			
			default:
				err = ERR_COMMANDLINE;
		}
	}
	
	if (err != ERR_NO) {
		printf("Commandline FlashProgrammer OPTION [FILE]\n");
		printf("   -P1   1.FILECRC, 2.DEVICE ERASE, 3.DEVICE WRITE, 4.DEVICE CRC\n");
		printf("   -P2   1.FILECRC, 2.DEVICE WRITE, 3.DEVICE CRC\n");
		printf("   -W    DEVICE WRITE\n");
		printf("   -E    DEVICE ERASE\n");
		printf("   -F    FILE CRC\n");
		printf("   -C    DEVICE CRC\n");
		printf("   -R    DEVICE READ\n");
		
		exit(1);
	}
	
	
	// Configure FPGA
	if ((err = configureFpga()) != ERR_NO) {
		printf("Error configuring FPGA: %d\n", err);
		exit(1);
	} else {
		printf("FPGA successfully configured\n");
	}
	
	
	// allocate memory
	if ((pBufferSave = (BYTE*) malloc(0x8004)) == NULL) {
		err = ERR_MEMORY;
	}
	pBuffer = pBufferSave;
	
	// init hardware
	if (err == ERR_NO) {
		err = hwInit();
	}
	
	// activate power and identify FlashCard/FlashSIMM
	if (err == ERR_NO) {
		hwPowerOn(1);
		hwVPPOn(1);
		
		err = flashGetDevicetype();
	} else {
		printf("**ERROR: %d\n", err);
		exit(1);
	}


	// main functions (Read/Write/Erase/...)
	if ((job & READ) != 0) {
		if (err == ERR_NO) {
			printf("JOB = READ\n");
			err = flashRead(filename);
		}
	}
	if ((job & FILECRC) != 0) {
		if (err == ERR_NO) {
			printf("JOB = FILECRC\n");
			err = flashFileCRC(filename, &filecrcsum);
		}
		if (err == ERR_NO) {
			printf("File CRC-SUM = %04X \n", filecrcsum);
		}
	}
	if ((job & ERASE) != 0) {
		if (err == ERR_NO) {
			printf("JOB = ERASE\n");
			err = flashErase();
		}
	}
	if ((job & WRITE) != 0) {
		if (err == ERR_NO) {
			printf("JOB = WRITE\n");
			err = flashWriteFromBinfile(filename);
		}
	}
	if ((job & FLASHCRC) != 0) {
		if (err == ERR_NO) {
			printf("JOB = FLASHCRC\n");
			err = flashReadCRC(&flashcrcsum, TRUE);
		}
		if (err == ERR_NO || err == ERR_CRC) {
			printf("CRC-SUM-SOLL = %04X  -  CRC-SUM-IST %04X\n", filecrcsum, flashcrcsum);
			if (filecrcsum != flashcrcsum) {
				err = ERR_CRC;
			}
		}
	}
	if ((job & FLASHCRCEND) != 0) {
		if (err == ERR_NO) {
			printf("JOB = FLASHCRCEND\n");
			err = flashReadCRC(&flashcrcsum, FALSE);
		}
		if (err == ERR_NO || err == ERR_CRC) {
			printf("CRC-SUM-IST = %04X\n", flashcrcsum);
		}
	}
	
	hwCleanup();
	
	if (err != ERR_NO) {
		printf("ERROR: %d\n", err);
	}
	else {
		printf("finished\n");
	}
}
